/*++
/* NAME
/*	memdecay
/* SUMMARY
/*	memory decay tester
/* SYNOPSIS
/*	\fBmemdecay\fR (anonymous memory)
/*
/*	\fBmemdecay -b\fR \fIbacking-file\fR (cached existing file data)
/*
/*	\fBmemdecay -b\fR \fIbacking-file\fR \fB-u\fR (cached deleted file)
/* DESCRIPTION
/*	This program fills a bunch of memory with the same unique but
/*	repeating pattern, deallocates the memory, and repeatedly scans
/*	/dev/mem for "evidence" from the filled memory until enough of it
/*	has disappeared, or until the number of memory scans reaches
/*	some limit.
/*
/*	Memory filling is done in a child process that exits before the
/*	parent process start the memory scan.
/*
/*	By default, the program uses anonymous memory that is backed by
/*	swap space. This simulates the decay of private memory after a
/*	process exits.
/*
/*	After each memory scan the program outputs one line with a time stamp,
/*	the number of kbytes in pages that contain fully intact data, and the
/*	number of intact kbytes in memory pages that were damaged by less
/*	than 50%.
/*
/*	The size options below understand the \fBk\fR (kilo) \fBM\fR (Mega)
/*	and \fBG\fR (Giga) suffixes.
/*
/*	Options
/* .IP "\fB-b \fIbacking-file\fR"
/*	Create the named file and use it as backing store for the filled
/*	memory. This simulates the decay of cached memory from existing
/*	files. See also the \fB-u\fR option below (unlink before memory scan).
/* .IP "\fB-d \fIloop-delay\fR (default: 5)"
/*	Repeat the memory scan every \fIloop-delay\fR seconds.
/* .IP "\fB-f \fIfill-size\fR (default: 1M)"
/*	The amount of memory to be filled.
/* .IP "\fB-h \fImax-scans\fR (default: 1000)"
/*	Specify a higher bound on the number of memory scans.
/* .IP "\fB-l \fImin-scans\fR (default: 100)"
/*	Specify a lower bound on the number of memory scans.
/* .IP "\fB-p \fIpage-size\fR (default: 1k)"
/*	Use \fIpage-size\fR as the memory page size. Specify 0 to use the
/*	system page size.
/* .IP "\fB-s \fIscan-size\fR (default: 0)"
/*	Number of memory bytes to scan. Specify 0 in order to scan all memory.
/*	Specify the memory size if your /dev/mem does not properly report
/*	end of memory.
/* .IP "\fB-t \fIthreshold\fR (default: 10)"
/*	Exit when less than \fIthreshold\fR percent of the filled memory is
/*	left over.
/* .IP \fB-u\fR
/*	Remove the file specified with \fB-b\fR before the memory scan
/*	starts. This simulates the decay of cached data from removed
/*	files. See also the -b option above (use mempory mapped file).
/* .IP \fB-v\fR
/*	Make the program more verbose; use this to see the memory map.
/* BUGS
/*	This program has a false hit rate of 1/256. It should be
/*	used only for short-term memory decay studies. The program
/*	over-estimates the amount of surviving memory when it reaches
/*	less than a few percent of the initial amount.
/*
/*	On FreeBSD you must specify the size of physical memory with the
/*	\fB-s\fR option, because /dev/mem does not report end of file.
/*
/*	On Solaris, memory has holes. Memory is located by probing the
/*	entire address range, which is slow. When no memory size is
/*	specified on the command line, or a too large memory size, the
/*	program will request to be re-run with the \fB-s\fR option.
/* LICENSE
/* .ad
/* .fi
/*      The IBM Public License must be distributed with this software.
/* AUTHOR
/*	Wietse Venema
/*	IBM T.J. Watson Research
/*	P.O. Box 704
/*	USA
/*--*/

/* System libraries. */

#include <sys/types.h>
#include <sys/mman.h>
#include <time.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>

#ifdef SUNOS5
#include <errno.h>
#endif

#ifndef MAP_NOCORE
#define MAP_NOCORE	0
#endif

/* Application-specific. */

#include "convert_size.h"
#include "make_template.h"
#include "error.h"

 /*
  * Default settings.
  */
#define DEF_LOOP_DELAY		5
#define DEF_FILL_SIZE		(1024*1024)	/* 1M */
#define DEF_PAGE_SIZE		1024	/* 1k */
#define DEF_SCAN_SIZE		0	/* all memory */
#define DEF_MATCH_THRESH	10	/* 10% */
#define DEF_MIN_SCANS		100
#define DEF_MAX_SCANS		1000

#define FLAG_UNLINK_BEFORE_SCAN	(1<<0)

#define PATH_DEV_MEM		"/dev/mem"

 /*
  * Memory map management.
  */
typedef struct {
    size_t  start;
    size_t  end;
}       MEM_MAP;

typedef struct {
    int     used;
    int     size;
    MEM_MAP *map;
}       MEM_INFO;

/* usage - complain, explain, and terminate */

static void usage(const char *why)
{
    if (why)
	remark("%s", why);
    error("usage: %s [options]\n"
	  "  -b backing-file-name    (use file as mmap()ed backing store)\n"
	  "  -d scan-loop-delay      (default: %d)\n"
	  "  -f memory-fill-size     (default: %ld)\n"
	  "  -h max-scan-count       (default: %ld)\n"
	  "  -l min-scan-count       (default: %ld)\n"
	  "  -p memory-page-size"
	  "     (default %ld, 0 = use system page size)\n"
	  "  -s memory-scan-size     (default %ld, 0 = scan all memory)\n"
	  "  -t match-treshold  "
	  "     (memory scan exit threshold, default %d%%)\n"
	  "  -u                 "
	  "     (unlink mmap()ed backing store before memory scan)",
	  "  -v                 "
	  "     (verbose mode for debugging)",
	  progname, DEF_LOOP_DELAY, (long) DEF_FILL_SIZE,
	  (long) DEF_MAX_SCANS, (long) DEF_MIN_SCANS,
	  (long) DEF_PAGE_SIZE, (long) DEF_SCAN_SIZE,
	  DEF_MATCH_THRESH);
}

/* fill_memory - fill a bunch of memory */

static void fill_memory(char *template, size_t fill_size, size_t page_size,
			        const char *backing_file, int flags)
{
    int     fd;
    int     mmap_flags;
    void   *ptr;
    void   *cp;

    /*
     * Allocate one chunk of memory, either anonymous memory that is backed
     * by swap space, or memory that is backed by a regular file. If it's a
     * regular file, extend it to the proper length before mapping it.
     */

#define NO_FILE	(-1)
#define ANYWHERE_YOU_LIKE	((void *) 0)
#define BITSOURCE		"/dev/zero"

    if (backing_file != 0) {
	if ((fd = open(backing_file, O_RDWR | O_CREAT | O_TRUNC, 0600)) < 0)
	    error("open mmap() backing file %s: %m", backing_file);
	if (lseek(fd, (off_t) fill_size - 1, SEEK_SET) < 0)
	    error("lseek %s: %m", backing_file);
	if (write(fd, "", 1) != 1)
	    error("write %s: %m", backing_file);
	mmap_flags = MAP_SHARED | MAP_NOCORE;
    } else {
#ifdef MAP_ANON
	fd = NO_FILE;
	mmap_flags = MAP_PRIVATE | MAP_NOCORE | MAP_ANON;
#else
	if ((fd = open(BITSOURCE, O_RDWR, 0)) < 0)
	    error("open %s: %m", BITSOURCE);
	mmap_flags = MAP_SHARED | MAP_NOCORE;
#endif
    }
    ptr = mmap(ANYWHERE_YOU_LIKE, fill_size, PROT_READ | PROT_WRITE,
	       mmap_flags, fd, (off_t) 0);
    if (ptr == MAP_FAILED)
	error("mmap failed: %m");

    /*
     * Paranoia: make sure that the memory is page aligned.
     */
    if ((((unsigned long) ptr) % page_size) != 0)
	error("mmap() result is not page aligned, bailing out");

    /*
     * Fill the memory with repeated pattern.
     */
    for (cp = ptr; cp < ptr + fill_size; cp += page_size)
	memcpy(cp, template, page_size);

    /*
     * Update backing store, if applicable.
     */
    if (backing_file != 0) {
	if (fsync(fd) < 0)
	    error("fsync mmap() backing file %s: %m", backing_file);
	if (close(fd) < 0)
	    error("close mmap() backing file %s: %m", backing_file);
    }

    /*
     * Unlink the mapped file, if applicable.
     */
    if ((flags & FLAG_UNLINK_BEFORE_SCAN) != 0 && backing_file != 0)
	if (unlink(backing_file) < 0)
	    error("unlink mmap() backing file %s: %m", backing_file);

    /*
     * Unmap the memory, so that the decay can begin.
     */
    if (munmap(ptr, fill_size) < 0)
	error("unmap memory: %m");
}

/* compare - compare two pieces of data */

static int compare(const char *a, const char *b, size_t len)
{
    size_t  same = 0;

    while (len-- > 0)
	same += (*a++ == *b++);
    return (same);
}

/* scan_memory - scan memory for "evidence" from memory fill pass */

static void scan_memory(int mem_fd, char *template, MEM_INFO * mem_info,
			        size_t page_size, char *buffer,
			        size_t loop_delay, size_t threshold,
			        size_t min_scans, size_t max_scans)
{
    size_t  parts;
    size_t  hits;
    size_t  where;
    size_t  same;
    size_t  scan;
    MEM_MAP *mp;

    /*
     * Scan memory for pages that match the template.
     */
    printf("time %ld start scan\n", (long) time((time_t *) 0));

    for (scan = 1; /* test at end */ ; scan++) {
	parts = hits = 0;
	for (mp = mem_info->map; mp < mem_info->map + mem_info->used; mp++) {
	    if (lseek(mem_fd, (off_t) mp->start, SEEK_SET) < 0)
		error("lseek %s: %m", PATH_DEV_MEM);
	    for (where = mp->start; where < mp->end; where += page_size) {
		if (read(mem_fd, buffer, page_size) != page_size)
		    error("read error (%m) at offset 0x%lx bytes, bailing out",
			  (unsigned long) where);
		if ((same = compare(template, buffer, page_size)) == page_size)
		    hits++;
		if (same > page_size / 2)
		    parts += same;
	    }
	}
	printf("time %ld     %ld kb full match     %ld kb partial match\n",
	       (long) time((time_t *) 0),
	       (long) hits * page_size / 1024,
	       (long) parts / 1024);
	fflush(stdout);
	if (hits * page_size < threshold && scan >= min_scans)
	    break;

	/*
	 * Prepare for another iteration.
	 */
	if (scan >= max_scans)
	    break;
	sleep(loop_delay);
    }
}

/* find_memory - locate memory blocks between holes */

static MEM_INFO *find_memory(int mem_fd, char *buffer,
			             size_t scan_size, size_t page_size)
{
    size_t  where;
    size_t  count;
    ssize_t read_count;
    int     region_count;
    MEM_INFO *mem_info;
    int     in_memory = 0;

    /*
     * Allocate and initialize the memory map.
     */
    if ((mem_info = (MEM_INFO *) malloc(sizeof(MEM_INFO))) == 0)
	error("no memory for memory map: %m");
    mem_info->used = 0;
    mem_info->size = 10;
    mem_info->map = (MEM_MAP *) malloc(mem_info->size * sizeof(MEM_MAP));
    if (mem_info->map == 0)
	error("no memory for memory map: %m");

    /*
     * Look for memory regions. Increment the loop variable at the loop end,
     * so we can skip over holes in Solaris memory.
     */
    for (where = 0, count = 0; scan_size == 0 || count < scan_size; /* increment at end */ ) {
	if ((read_count = read(mem_fd, buffer, page_size)) < 0) {

	    /*
	     * Update the memory map's idea of where the last region ends.
	     */
	    if (in_memory) {
		mem_info->map[mem_info->used].end = where;
		in_memory = 0;
		mem_info->used++;
	    }

	    /*
	     * Skip over holes in Solaris memory.
	     */
#ifdef SUNOS5
	    if (errno == EFAULT) {
		if (lseek(mem_fd, page_size, SEEK_CUR) < 0)
		    error("please re-run this command with: -s %lu",
			  (unsigned long) count);
		where += page_size;
		continue;
	    }
#endif
	    error("memory read error (%m) after %lu bytes, bailing out",
		  (long) count);
	} else if (read_count != page_size) {
	    break;
	} else {

	    /*
	     * Update the memory map's idea of where this memory region
	     * starts.
	     */
	    if (in_memory == 0) {
		in_memory = 1;
		if (mem_info->used >= mem_info->size) {
		    mem_info->size *= 2;
		    mem_info->map = (MEM_MAP *)
			realloc(mem_info->map, region_count * sizeof(MEM_MAP));
		    if (mem_info->map == 0)
			error("no memory for memory map: %m");
		}
		mem_info->map[mem_info->used].start = where;
	    }
	    count += page_size;
	    where += page_size;
	}
    }

    /*
     * Update the memory map's idea of where the last region ends.
     */
    if (in_memory) {
	mem_info->map[mem_info->used].end = where;
	in_memory = 0;
	mem_info->used++;
    }

    /*
     * Debugging info.
     */
    if (verbose) {
	MEM_MAP *mp;
	size_t  total = 0;

	printf("Memory map (start, end) pairs:\n");
	for (mp = mem_info->map; mp < mem_info->map + mem_info->used; mp++) {
	    printf("0x%lx	0x%lx (%lu)\n",
		   (unsigned long) mp->start,
		   (unsigned long) mp->end,
		   (unsigned long) (mp->end - mp->start));
	    total += mp->end - mp->start;
	}
	printf("Total of %lu bytes\n", total);
	fflush(stdout);
    }
    return (mem_info);
}

/* main - main program */

int     main(int argc, char **argv)
{
    char   *backing_file = 0;
    size_t  loop_delay = DEF_LOOP_DELAY;
    size_t  fill_size = DEF_FILL_SIZE;
    size_t  min_scans = DEF_MIN_SCANS;
    size_t  max_scans = DEF_MAX_SCANS;
    size_t  page_size = DEF_PAGE_SIZE;
    size_t  scan_size = DEF_SCAN_SIZE;
    size_t  threshold = 10;
    int     flags = 0;
    int     ch;
    time_t  seed;
    char   *template;
    size_t  excess;
    double  factor;
    pid_t   child_pid;
    int     child_status;
    int     mem_fd;
    MEM_INFO *mem_info;
    char   *buffer;

    progname = argv[0];

    /*
     * Parse JCL.
     */
    while ((ch = getopt(argc, argv, "b:d:f:h:l:p:s:t:uv")) > 0) {
	switch (ch) {
	case 'b':
	    backing_file = optarg;
	    break;
	case 'd':
	    if ((loop_delay = convert_size(optarg)) == -1)
		usage("bad loop delay");
	    break;
	case 'f':
	    if ((fill_size = convert_size(optarg)) == -1)
		usage("bad memory fill size");
	    break;
	case 'h':
	    if ((max_scans = convert_size(optarg)) == -1)
		usage("bad maximal scan count");
	    break;
	case 'l':
	    if ((min_scans = convert_size(optarg)) == -1)
		usage("bad minimal scan count");
	    break;
	case 'p':
	    if ((page_size = convert_size(optarg)) == -1)
		usage("bad memory size");
	    break;
	case 's':
	    if ((scan_size = convert_size(optarg)) == -1)
		usage("bad memory scan size");
	    break;
	case 't':
	    if ((threshold = convert_size(optarg)) == -1)
		usage("bad match threshold");
	    break;
	case 'u':
	    flags |= FLAG_UNLINK_BEFORE_SCAN;
	    break;
	case 'v':
	    verbose++;
	    break;
	default:
	    usage((char *) 0);
	    break;
	}
    }
    if (optind != argc)
	usage((char *) 0);

    /*
     * Sanity checks.
     */
    if (page_size == 0) {
	page_size = getpagesize();
	printf("%s: using system page size %ld\n", argv[0], (long) page_size);
    }
    if (fill_size > 0 && fill_size < page_size)
	error("fill size %ld is smaller than page size %ld, bailing out",
	      (long) fill_size, (long) page_size);
    if (loop_delay < 1)
	error("loop delay %ld is less than 1, bailing out",
	      (long) loop_delay);
    if (min_scans > max_scans)
	error("min memory scan count %ld is larger than max scan count %ld",
	      (long) min_scans, (long) max_scans);

    /*
     * Everything must be a multiple of the page size.
     */
    if (fill_size > 0 && (excess = (fill_size % page_size)) != 0) {
	remark("fill size %ld is not multiple of page size %ld, rounding down",
	       (long) fill_size, (long) page_size);
	fill_size -= excess;
    }
    if (scan_size > 0 && (excess = (scan_size % page_size)) != 0) {
	remark("scan size %ld is not multiple of page size %ld, rounding down",
	       (long) scan_size, (long) page_size);
	scan_size -= excess;
    }

    /*
     * Compute our initial filler template.
     */
    seed = time((time_t *) 0);
    template = make_template((char *) &seed, sizeof(seed), page_size);

    /*
     * Find out where the holes in memory are so that we can skip them
     * efficiently. The memory map is a null-terminated list of (start page,
     * number of pages) tuple that specify where memory was found.
     */
    if ((mem_fd = open(PATH_DEV_MEM, O_RDONLY, 0)) < 0)
	error("open %s: %m", PATH_DEV_MEM);
    if ((buffer = malloc(page_size)) == 0)
	error("no buffer memory: %m");
    mem_info = find_memory(mem_fd, buffer, scan_size, page_size);

    /*
     * Allocate memory, and fill each page with the same pattern. Just to be
     * on the safe side we fill the memory as a child process that terminates
     * before the memory scan starts.
     */
    if ((child_pid = fork()) == -1)
	error("cannot fork a child process: %m");
    if (child_pid == 0) {
	fill_memory(template, fill_size, page_size, backing_file, flags);
	exit(0);
    } else {
	wait(&child_status);
    }

    if (child_status != 0)
	error("child failed: exit status %d", child_status);

    /*
     * Scan memory.
     */
    factor = threshold / 100.0;
    scan_memory(mem_fd, template, mem_info, page_size, buffer, loop_delay,
		(int) (fill_size * factor), min_scans, max_scans);

    /*
     * Clean up.
     */
    if (close(mem_fd) < 0)
	error("close %s: %m", PATH_DEV_MEM);

    free(buffer);

    exit(0);
}
