 /*
  * TCT drop-off daemon, run from inetd under tcpd.
  * 
  * Usage: tct-receive /directory/name
  * 
  * The data from each client connection is dumped to a file with name
  * clientaddr-yyyymmdd.hhmmss-suffix, where the suffix avoids file name
  * collisions.
  * 
  * Author: Wietse Venema, IBM T.J.Watson Research.
  * 
  * The IBM public license must be distributed with this software.
  */
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <stdlib.h>
#include <unistd.h>
#include <syslog.h>
#include <stdio.h>
#include <string.h>
#include <fcntl.h>
#include <errno.h>
#include <signal.h>
#include <arpa/inet.h>
#include <time.h>

 /*
  * How often a client can drop off a file in a second.
  */
#define MAX_TRIES	10

 /*
  * How long a connection can be idle.
  */
#define TIMEOUT		1000

static struct sockaddr_in sin;

void    timeout(int sig)
{
    syslog(LOG_ERR, "client %s: timeout", inet_ntoa(sin.sin_addr));
    exit(1);
}

int     main(int argc, char **argv)
{
    int     len = sizeof(sin);
    char    path_buf[BUFSIZ];
    char    date_buf[BUFSIZ];
    char    data_buf[8192];
    char   *cp;
    int     suffix;
    int     fd;
    int     count;
    time_t  now;

    /*
     * Sane umask.
     */
    umask(022);

    /*
     * Don't get stuck.
     */
    signal(SIGALRM, timeout);

    /*
     * Get process basename.
     */
    if ((cp = strrchr(argv[0], '/')) != 0)
	argv[0] = cp + 1;

    /*
     * Log as daemon.
     */
    openlog(argv[0], LOG_PID, LOG_DAEMON);

    /*
     * Sanity check.
     */
    if (argc != 2) {
	syslog(LOG_ERR, "Usage: %s directory", argv[0]);
	sleep(1);
	exit(1);
    }

    /*
     * Look up client name.
     */
    if (getpeername(0, (struct sockaddr *) & sin, &len) < 0) {
	syslog(LOG_ERR, "getpeername: %m");
	sleep(1);
	exit(1);
    }

    /*
     * Open a unique drop file.
     */
    time(&now);
    strftime(date_buf, sizeof(date_buf), "%Y%m%d.%H%M%S",  gmtime(&now));
    for (suffix = 0; /* void */ ; suffix++) {
	if (suffix >= MAX_TRIES) {
	    syslog(LOG_ERR, "%s: too many attempts", inet_ntoa(sin.sin_addr));
	    sleep(1);
	    exit(1);
	}
	sprintf(path_buf, "%s/%s-%s-%d", argv[1], inet_ntoa(sin.sin_addr),
		date_buf, suffix);
	if ((fd = open(path_buf, O_WRONLY | O_CREAT | O_EXCL, 0644)) > 0) {
	    syslog(LOG_NOTICE, "client %s: opened file %s",
		   inet_ntoa(sin.sin_addr), path_buf);
	    break;
	}
	if (errno != EEXIST) {
	    syslog(LOG_ERR, "open %s: %m", path_buf);
	    sleep(1);
	    exit(1);
	}
    }

    /*
     * Copy data to file.
     */
    for (;;) {
	alarm(TIMEOUT);
	if ((count = read(0, data_buf, sizeof(data_buf))) == 0)
	    break;
	if (count < 0) {
	    syslog(LOG_ERR, "client %s: read error: %m",
		   inet_ntoa(sin.sin_addr));
	    sleep(1);
	    exit(1);
	}
	if (write(fd, data_buf, count) < 0) {
	    syslog(LOG_ERR, "client %s: file %s: write error: %m",
		   inet_ntoa(sin.sin_addr), path_buf);
	    sleep(1);
	    exit(1);
	}
    }
    if (close(fd) != 0) {
	syslog(LOG_ERR, "client %s: file %s: write error: %m",
	       inet_ntoa(sin.sin_addr), path_buf);
	sleep(1);
	exit(1);
    }
    exit(0);
}
