 /*
  * TCT drop-off client
  * 
  * Usage: tct-send host-ip-number port-number
  * 
  * Copies stdin to the specified tct-receive server.
  * 
  * Author: Wietse Venema, IBM T.J.Watson Research.
  * 
  * The IBM public license must be distributed with this software.
  */
#include <sys/types.h>
#include <sys/socket.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <netdb.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <signal.h>
#include <string.h>

#ifndef INADDR_NONE
#define INADDR_NONE 0xFFFFFFFF
#endif

 /*
  * How long a connection can be idle.
  */
#define TIMEOUT		1000

static struct sockaddr_in sin;

static void usage(const char *myname)
{
    fprintf(stderr, "Usage: %s host-ip-number port-number\n", myname);
    sleep(1);
    exit(1);
}

void    timeout(int sig)
{
    fprintf(stderr, "sending to server %s: timeout", inet_ntoa(sin.sin_addr));
    sleep(1);
    exit(1);
}

static void perrorexit(const char *s)
{
    perror(s);
    sleep(1);
    exit(1);
}

int     main(int argc, char **argv)
{
    char    buf[8192];
    int     sock;
    long    addr;
    unsigned port;
    int     rcount;
    int     wcount;
    char   *cp;

    /*
     * Don't get stuck.
     */
    signal(SIGALRM, timeout);

    /*
     * Pipe errors become write errors.
     */
    signal(SIGPIPE, SIG_IGN);

    /*
     * Get process basename.
     */
    if ((cp = strrchr(argv[0], '/')) != 0)
	argv[0] = cp + 1;

    /*
     * Parse JCL.
     */
    if (argc != 3)
	usage(argv[0]);
    if ((addr = inet_addr(argv[1])) == INADDR_NONE)
	usage(argv[0]);
    if ((port = htons(atoi(argv[2]))) == 0)
	usage(argv[0]);

    /*
     * Connect.
     */
    if ((sock = socket(AF_INET, SOCK_STREAM, 0)) < 0)
	perrorexit("socket");
    memset((char *) &sin, 0, sizeof(sin));
    sin.sin_family = AF_INET;
    sin.sin_addr.s_addr = addr;
    sin.sin_port = port;
    if (connect(sock, (struct sockaddr *) & sin, sizeof(sin)) < 0)
	perrorexit("connect");

    /*
     * Send data.
     */
    for (;;) {
	alarm(TIMEOUT);
	if ((rcount = read(0, buf, sizeof(buf))) < 0)
	    perrorexit("read");
	if (rcount == 0)
	    break;
	for (cp = buf; rcount > 0; /* void */ ) {
	    if ((wcount = write(sock, cp, rcount)) < 0)
		perrorexit("write");
	    rcount -= wcount;
	    cp += wcount;
	}
    }
    exit(0);
}
