#++
# NAME
#	capture 1
# SUMMARY
#	capture and checksum command output
# SYNOPSIS
#	require "capture.pl";
#
#	@handles = &capture_open($destination, $output_file)
#
#	&capture_write(@handles, list...)
#
#	&capture_execute(@handles, @command)
#
#	&capture_close(@handles)
# DESCRIPTION
#	This module hides the complexity of capturing command output
#	either locally or remotely, and of MD5 hashing data on the fly.
#
#	The code manipulates a pair of streams, a data stream for actual 
#	data, for data, and an MD5 stream for computing the MD5 checksum.
#
#	capture_open() opens a data and MD5 stream. $output_file
#	specifies the relative pathname of an output file.
#
#	If $destination looks like "|command", then the data stream
#	pipes into "|command", and the content is prefixed with
#	"#!$output_file". The MD5 stream pipes into "|$MD5|command",
#	the MD5 stream content is prefixed with a header of: 
#	"#!$output_file.md5". 
#
#	Otherwise, output is written to $destination/$output_file, and 
#	the MD5 checksum is written to $destination/$output_file.md5.
#
#	capture_write() writes its argument list to both the data stream
#	and the MD5 stream. This can be used for data headers or for actual
#	data content.
#
#	capture_execute() executes the specified command, and copies its
#	output to both the data stream and the MD5 stream.
#
#	capture_close() closes the data stream and the MD5 stream.
#
#	Arguments:
# .IP destination
#	Destination directory name, or "|command".
# .IP output_file
#	Relative pathname of output file.
# .IP handles
#	Result from capture_open() that must be passed to all other routines
#	in this module.
# .IP list
#	List of parameters, suitable for use with the print routine.
# .IP command
#	The command whose output is captured and on the fly MD5 checksummed.
# LICENSE
#	This software is distributed under the IBM Public License.
# AUTHOR(S)
#	Wietse Venema
#	IBM T.J. Watson Research
#	P.O. Box 704, Yorktown Heights, NY 10598, USA
#--

#
# This is so that we can allocate file handless on the fly instead of having
# to use global file handles names, which is a bad idea for large programs.
#
use IO::File;

# capture_open - open output channel and checksum channel

sub capture_open()
{
    my($destination, $output_file) = @_;
    my($data_handle, $md5_handle);

    $data_handle = new IO::File;
    $md5_handle = new IO::File;

    if ($destination =~ /^\|(.*)/) {
	$command = $1;
	&log_item("piping into command \"$command\" (output)\n");
	open($data_handle, "|$command")
	    || die "capture: Cannot execute command \"$command\": !$\n";
	print $data_handle "#! $output_file\n";

	&log_item("piping into command \"$MD5|$command\" (MD5 hash)\n");
	open ($md5_handle, "|(echo '#! $output_file.md5'; $MD5)|$command")
	    || die "capture: Cannot execute command \"$MD5|$command\": !$\n";
    } else {
	$output_file = "$destination/$output_file";
	&log_item("opening output file \"$output_file\" (output)\n");
	open($data_handle, ">$output_file")
	    || die "capture: Cannot create file $output_file: $!\n";

	&log_item("piping into command \"$MD5 >$output_file.md5\" (MD5 hash)\n");
	open($md5_handle, "|$MD5 >$output_file.md5")
	    || die "capture: Cannot execute command \"$MD5 >$output_file.md5\": $!\n";
    }
    return ($data_handle, $md5_handle);
}

# capture_write - append text to capture streams

sub capture_write()
{
    my($data_handle) = shift(@_);
    my($md5_handle) = shift(@_);

   # Copy to both the data stream and to the MD5 stream.

    print $data_handle @_;
    print $md5_handle @_;
}

# capture_execute - copy command output to capture streams

sub capture_execute()
{
    my($data_handle) = shift(@_);
    my($md5_handle) = shift(@_);
    my(@command) = @_;

    &pipe_command(COMMAND, @command, "-|")
	|| die "capture_execute: cannot execute @command: $!\n";
     while(<COMMAND>) {
	print $data_handle $_;
	print $md5_handle $_;
    }
    close(COMMAND);
    die "capture: @command terminated with status $?\n" if $?;
}

# capture_close close capture streams

sub capture_close()
{
    my($data_handle, $md5_handle) = @_;

    close($data_handle) || die "capture_close: write error: $! $?\n";
    close($md5_handle) || die "capture_close: write error: $! $?\n";
}

# Scaffolding...

if ($running_under_grave_robber) {
    require "command.pl";
    require "logger.pl";
    require "tm_misc.pl";
} else {
    $running_under_grave_robber = 1;
    require "conf/coroner.cf";
    require "command.pl";
    require "logger.pl";
    require "tm_misc.pl";
    @handles = &capture_open("|cat -vn", "example");
    #@handles = &capture_open("/tmp", "example");
    &capture_write(@handles, "this is from capture_write\n");
    &capture_execute(@handles, "echo", "this is from capture_execute");
    &capture_close(@handles);
}

1;
