/*++
/* NAME
/*	fssum 1
/* SUMMARY
/*	file system block checksummer
/* SYNOPSIS
/* .ad
/* .fi
/*	\fBfssum\fR [\fB-HvV\fR] [\fB-b \fIblocksize\fR] [\fB-f \fIfstype\fR]
/*		[\fB-h \fIhash_size\fR] \fIdevice\fR [\fIstart-stop\fR ...]
/* DESCRIPTION
/*	\fBfssum\fR opens the named \fIdevice\fR and computes MD5
/*	hashes over blocks of data.
/*	By default, \fBfssum\fR examines the entire \fIdevice\fR.
/*	See the OUTPUT FORMAT section below for output format details.
/*
/* 	Arguments:
/* .IP "\fB-b \fIblocksize\fR"
/*	The number of bytes over which an MD5 hash is computed.
/*	The default is 1024 bytes. This block size is unrelated to
/*	the block size used internally by the file system.
/* .IP \fB-H\fR
/*	Do not prepend a time machine header to the data (see the
/*	OUTPUT FORMAT section below).
/* .IP "\fB-h \fIhash_size\fR"
/*	The number of output bits per hash. The number must be a multiple
/*	of 8. The default number of bits per hash is 16 bits.
/*	The maximal number of bits is 128 (i.e. the full output
/*	from the MD5 algorithm).
/* .IP "\fB-f\fI fstype\fR"
/*	Specifies the file system type. The default file system type
/*	is system dependent. With most UNIX systems the default type
/*	is \fBufs\fR (Berkeley fast file system). With Linux the default
/*	type is \fBext2fs\fR (second extended file system).
/* .IP \fB-v\fR
/*	Turn on verbose mode, output to stderr.
/* .IP \fB-V\fR
/*	Turn on verbose mode, output to stdout.
/* .IP \fIdevice\fR
/*	Disk special file, or regular file containing a disk image.
/*	On UNIX systems, raw mode disk access may give better performance
/*	than block mode disk access.  LINUX disk device drivers support
/*	only block mode disk access.
/* .IP "\fIstart-stop\fR ..."
/*	Examine the specified block number or number range. Either the
/*	\fIstart\fR, the \fIstop\fR, or the \fI-stop\fR may be omitted.
/*	With the FFS file system, block numbers start at zero.
/*	With the LINUX ext2fs file system, the start block number must be >= 1.
/* OUTPUT FORMAT
/* .ad
/* .fi
/*	The output format is in time machine format, as described in
/*	tm-format(5).
/*
/*	For every \fIdevice\fR, the output begins with a two-line header that
/*	describes the data origin, device name and hash block size.
/*
/*	The information is followed by a one-line header that lists the names
/*	of the data attributes that make up the remainder of the output:
/* .IP \fBstatus\fR
/*	Status of the block: \fBa\fR (allocated), \fBf\fR (free),
/*	\fBo\fR (other, either a bitmap or inode block).
/* .IP \fBhash\fR
/*	The MD5 checksum.
/* SEE ALSO
/*	tm-format(5), time machine file format.
/* BUGS
/*	\fBfssum\fR should support more file system types. Right now, support
/*	is limited to \fBext2fs\fR when built on Linux, and \fBufs\fR when
/*	built on Solaris and BSD systems.
/*
/*	Due to programmer laziness, the file system block size must be a
/*	multiple of the hash block size. With a default hash block size
/*	of 1024 bytes, this is not a problem on any supported system.
/* LICENSE
/*	This software is distributed under the IBM Public License.
/* AUTHOR(S)
/*	Wietse Venema
/*	IBM T.J. Watson Research
/*	P.O. Box 704
/*	Yorktown Heights, NY 10598, USA
/*--*/

#include "fs_tools.h"
#include "error.h"
#include "split_at.h"
#include "mymalloc.h"
#include "global.h"
#include "md5.h"

#define DEFAULT_BLOCK_SIZE      1024	/* bytes */
#define DEFAULT_HASH_SIZE       16	/* bits */
#define MD5_HASH_LENGTH         16	/* characters */

FILE   *logfp;

static unsigned long block_size = DEFAULT_BLOCK_SIZE;
static int hash_size = DEFAULT_HASH_SIZE;
static int prepend_header = 1;

/* atoblock - convert string to block number */

DADDR_T atoblock(const char *str)
{
    char   *cp;
    DADDR_T addr;

    if (*str == 0)
	return (0);
    addr = STRTOUL(str, &cp, 0);
    if (*cp || cp == str)
	error("bad block number: %s", str);
    return (addr);
}

/* usage - explain and terminate */

static void usage()
{
    error("usage: %s [-b block_size] [-f fstype] [-h hash_size] [-vV] device [block... ]",
	  progname);
}

/* print_header - print time machine header */

static void print_header(const char *device, FS_INFO * fs)
{
    char    hostnamebuf[BUFSIZ];
    unsigned long now;

    if (gethostname(hostnamebuf, sizeof(hostnamebuf) - 1) < 0)
	error("gethostname: %m");
    hostnamebuf[sizeof(hostnamebuf) - 1] = 0;
    now = time((time_t *) 0);

    /*
     * Identify table type and table origin.
     */
    printf("class|host|device|blocksize|start_time\n");
    printf("fssum|%s|%s|%lu|%lu\n", hostnamebuf, device, block_size, now);

    /*
     * Identify the fields in the data that follow.
     */
    printf("status|hash\n");
}

/* print_block - write data block to stdout */

static void print_block(DADDR_T addr, char *buf, int flags, char *ptr)
{
    FS_INFO *fs = (FS_INFO *) ptr;
    MD5_CTX md;
    unsigned char sum[MD5_HASH_LENGTH];
    static char hex[] = "0123456789abcdef";
    int     i;
    int     status;
    char   *cp;

    if (verbose)
	fprintf(logfp, "write block %lu\n", (ULONG) addr);

    status = ((flags & FS_FLAG_UNALLOC) ? 'f' :
	      (flags & FS_FLAG_META) ? 'o' :
	      (flags & FS_FLAG_ALLOC) ? 'a' :
	      '?');

    /*
     * For now, require that the file system block size is a multiple of the
     * hash block size.
     */
    for (cp = buf; cp < buf + fs->block_size; cp += block_size) {
	MD5Init(&md);
	MD5Update(&md, cp, block_size);
	MD5Final(sum, &md);

	putchar(status);
	putchar('|');
	for (i = 0; i < hash_size / 8; i++) {
	    putchar(hex[(sum[i] >> 4) & 0xf]);
	    putchar(hex[sum[i] & 0xf]);
	}
	putchar('\n');
    }
}

/* main - open file system, list block info */

int     main(int argc, char **argv)
{
    FS_INFO *fs;
    char   *start;
    char   *last;
    DADDR_T bstart;
    DADDR_T blast;
    int     ch;
    int     flags = FS_FLAG_UNALLOC | FS_FLAG_ALLOC | FS_FLAG_META;
    char   *fstype = DEF_FSTYPE;
    char   *cp;
    char   *device;

    progname = argv[0];

    while ((ch = getopt(argc, argv, "b:f:Hh:vV")) > 0) {
	switch (ch) {
	default:
	    usage();
	case 'b':
	    block_size = STRTOUL(optarg, &cp, 0);
	    if (*cp || cp == optarg || block_size <= 0)
		error("invalid -b option value: %s", optarg);
	    break;
	case 'f':
	    fstype = optarg;
	    break;
	case 'H':
	    prepend_header = 0;
	    break;
	case 'h':
	    if ((hash_size = atoi(optarg)) <= 0)
		error("invalid -h option value: %s", optarg);
	    if (hash_size % 8)
		error("-h option requires multiple of 8");
	    if (hash_size > 128)
		error("-h option value must not exceed 128");
	    break;
	case 'v':
	    verbose++;
	    logfp = stderr;
	    break;
	case 'V':
	    verbose++;
	    logfp = stdout;
	    break;
	}
    }

    if (optind >= argc)
	usage();

    device = argv[optind++];

    /*
     * Open the file system.
     */
    fs = fs_open(device, fstype);

    if (fs->block_size % block_size) {
	remark("File system block size is %d", (int) fs->block_size);
	error("For now, file system block size must be multiple of hash block size");
    }

    /*
     * Emit some time machine header information.
     */
    if (prepend_header)
	print_header(device, fs);

    /*
     * Output the named data blocks, subject to the specified restrictions.
     */
    if (optind < argc) {
	while ((start = argv[optind]) != 0) {
	    last = split_at(start, '-');
	    bstart = (*start ? atoblock(start) : fs->start_block);
	    blast = (!last ? bstart : *last ? atoblock(last) : fs->last_block);
	    fs->block_walk(fs, bstart, blast, flags, print_block, (char *) fs);
	    optind++;
	}
    }

    /*
     * Output all blocks, subject to the specified restrictions.
     */
    else {
	fs->block_walk(fs, fs->start_block, fs->last_block,
		       flags, print_block, (char *) fs);
    }
    fs->close(fs);
    exit(0);
}
